#!/bin/bash
set -eo pipefail

default_api_key_path=/var/opt/fortanix/em-agent/join-token
em_agent_conf_file=/etc/em-agent/em-agent.conf
TMP_DIR="/tmp/em-agent/"
EM_AGENT_PKG_NAME="em-agent-snp"

if [ "$EUID" -ne 0 ]; then
    run_as_root_msg="Please run this file as root"
    echo ${run_as_root_msg}
    exit 1
fi

function create_new_dir() {
    rm -rf $1
    mkdir -p $1
}

create_new_dir $TMP_DIR

function usage() {
    installer_usage="USAGE:    bash $0 <join-token>"
    echo -e "${installer_usage}"
}

function check_sev_snp_module() {
    local vendor_details=$(lscpu | grep "Vendor ID" | grep -i AMD)
    local cpu_details=$(grep -o 'sev_snp' /proc/cpuinfo | uniq)
    if [[ -n "$vendor_details" && -n "$cpu_details" ]]; then
        echo "* The hardware platform does have SEV SNP support" >&3
    else
        echo "* The hardware platform does not have SEV SNP support" >&3
        die
    fi
}

function check_kernel_module() {
    local kernel_details=$(modinfo vhost_vsock)
    if [[ -n "$kernel_details" ]]; then
        echo "* The hardware platform does have vhost_vsock kernel module" >&3
    else
        echo "* The hardware platform does not have vhost_vsock kernel module" >&3
        die
    fi
}

function append_line_if_not_exists() {
    grep -qxF "$1" "$2" || echo "$1" >> "$2"
}

function die() {
    echo "error - see log file $log_file for full details" >&3
    echo "Tail of the log file:" >&3
    tail -n 15 "$log_file" >&3
    exit 1
}

clean() {
    rm -rf $TMP_DIR || true
}

trap clean EXIT

while [ "$1" != "" ]; do
    case $1 in
        -h | --help )                usage
                                     exit
                                     ;;
        * )                  api_key="$1"
    esac
    shift
done

if [ -z "$api_key" ]; then
    join_token_missing_msg="Error: join token argument is missing. The join token is the way by which this node will authenticate itself to Enclave Manager. Please consult the INSTALLER_README.md file for how to get a join token from Enclave Manager."
    echo -e $join_token_missing_msg
    usage
    exit 1
fi

set -u
log_file=$(mktemp /var/log/node_agent_installer.XXXXXXXXX)
echo "This installation will be logged at $log_file"
exec 3>&1 4>&2
trap 'exec 2>&4 1>&3' 0 1 2 3
exec 1>"$log_file" 2>&1
set -x

# Try to read the API_KEY_PATH from /etc/em-agent/em-agent.conf if that file already exists from a previous
# installation. The em-agent.deb package won't overwrite that config file if it already exists.
if [ -f "$em_agent_conf_file" ]; then
    source "$em_agent_conf_file"
fi
api_key_path="${API_KEY_PATH:-$default_api_key_path}"

echo -n "* Installing join token to $api_key_path... " >&3
mkdir -p "$(dirname "$api_key_path")" || die
echo "$api_key" > "$api_key_path" || die
echo "done" >&3

check_kernel_module || die
check_sev_snp_module || die

echo -n "* Updating package list... " >&3
apt-get update || die
echo "done" >&3

install_packages_msg="Installing $EM_AGENT_PKG_NAME"
packages_to_install=("$EM_AGENT_PKG_NAME")

install_packages_msg="$install_packages_msg... "

echo -n "* Installing Fortanix apt repository... " >&3
APT_REPOSITORY="https://download.fortanix.com/linux/apt"
GPG_KEY="/usr/share/keyrings/fortanix-public.gpg"
GPG_SUM=a4b08ecf65595012cde191fb5600f645df0175ce0d2686293a670c515981e555
DISTRIBUTION=noble
apt update
apt install ca-certificates curl gpg -y
install -m 0755 -d /etc/apt/keyrings
if [[ ! -f "${GPG_KEY}" || "$(sha256sum "$GPG_KEY")" != "$GPG_SUM  $GPG_KEY" ]];then
    rm -f "${GPG_KEY}"
    curl -fsSL "$APT_REPOSITORY/fortanix.gpg" | gpg --dearmor -o ${GPG_KEY} 
    chmod a+r ${GPG_KEY}
fi
# Add the repository to Apt sources:
tee /etc/apt/sources.list.d/fortanix-private.sources <<EOF
Enabled: yes
Types: deb
URIs: ${APT_REPOSITORY}
Architectures: amd64
Suites: ${DISTRIBUTION}
Components: main
Signed-By: ${GPG_KEY}
EOF
apt update
echo "done" >&3

echo -n "$install_packages_msg" >&3
apt-get install -y "${packages_to_install[@]}"
echo "done" >&3

echo "Installation finished" >&3